<?php

declare(strict_types=1);

namespace App\Models;

use Exception;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class Module
 *
 * @property int id
 * @property string version
 * @property string name
 * @property int disabled
 * @property int updated_at
 * @property int created_at
 */
class Module extends BaseModel
{
    /**
     * Indicates if the model should be timestamped.
     */
    public $timestamps = false;

    /**
     * The attributes that aren't mass assignable.
     */
    protected $guarded = [];

    /**
     * Assets modules path
     */
    public string $assetsPath = '/assets/modules/';

    /**
     * Выполняет применение миграции
     */
    public function migrate(string $modulePath): void
    {
        $migrationPath = $modulePath . '/migrations';

        if (file_exists($migrationPath)) {
            Artisan::call('migrate', [
                '--force'    => true,
                '--realpath' => true,
                '--path'     => $migrationPath,
            ]);
        }
    }

    /**
     * Выполняет откат миграций
     */
    public function rollback(string $modulePath): void
    {
        $migrationPath = $modulePath . '/migrations';

        if (file_exists($migrationPath)) {
            $migrator = app('migrator');
            $nextBatchNumber = $migrator->getRepository()->getNextBatchNumber();
            $migrationNames = array_keys($migrator->getMigrationFiles($migrationPath));

            DB::table(config('database.migrations'))
                ->whereIn('migration', $migrationNames)
                ->update(['batch' => $nextBatchNumber]);

            Artisan::call('migrate:rollback', [
                '--force'    => true,
                '--realpath' => true,
                '--path'     => $migrationPath,
            ]);
        }
    }

    /**
     * Создает симлинки модулей
     */
    public function createSymlink(string $modulePath): void
    {
        $filesystem = new Filesystem();
        $originPath = public_path($this->getLinkName($modulePath));
        $modulesPath = $modulePath . '/resources/assets';

        if (! file_exists($modulesPath)) {
            return;
        }

        $filesystem->link($modulesPath, $originPath);
    }

    /**
     * Удаляет симлинки модулей
     */
    public function deleteSymlink(string $modulePath): void
    {
        $originPath = public_path($this->getLinkName($modulePath));

        if (! file_exists($originPath)) {
            return;
        }

        $filesystem = new Filesystem();
        $filesystem->delete($originPath);
    }

    /**
     * Получает название директории для симлинка
     */
    public function getLinkName(string $modulePath): string
    {
        return $this->assetsPath . Str::plural(strtolower(basename($modulePath)));
    }

    /**
     * Get enabled modules
     */
    public static function getEnabledModules(): array
    {
        try {
            $modules = self::query()->where('disabled', 0)->pluck('name')->all();
        } catch (Exception) {
            $modules = [];
        }

        return $modules;
    }
}
