<?php
// Включение отображения ошибок для отладки (убрать на продакшене)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Проверка, была ли сессия уже запущена
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config.php';
require_once 'includes/db.php';
require_once 'includes/auth.php';

// Проверка авторизации
require_login(); // Убедимся, что пользователь авторизован

global $pdo;
$user_id = $_SESSION['user_id'];

// Получаем данные текущего пользователя
$stmt = $pdo->prepare("SELECT id, login, name, city, website, about, avatar FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

if (!$user) {
    header('Location: index.php');
    exit;
}

// Генерация CSRF-токена
$csrf_token = generate_csrf_token(); // Используем функцию из includes/auth.php

// Обработка загрузки аватара
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_avatar'])) {
    // Проверка CSRF-токена
    if (!verify_csrf_token()) {
        $error = "Недействительный запрос. Попробуйте снова.";
    } else {
        if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['avatar'];
            $fileName = basename($file['name']);
            $fileType = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            $allowedTypes = ['jpg', 'jpeg', 'png', 'gif'];
            
            if (in_array($fileType, $allowedTypes)) {
                $newFileName = 'avatars/' . uniqid() . '.' . $fileType;
                move_uploaded_file($file['tmp_name'], $newFileName);
                
                // Удаляем старый аватар, если он есть
                if (!empty($user['avatar']) && file_exists($user['avatar'])) {
                    unlink($user['avatar']);
                }
                
                $stmt = $pdo->prepare("UPDATE users SET avatar = ? WHERE id = ?");
                $stmt->execute([$newFileName, $user_id]);
                header('Location: profile.php');
                exit;
            } else {
                $error = "Недопустимый формат файла. Используйте JPG, JPEG, PNG или GIF.";
            }
        } else {
            $error = "Ошибка загрузки файла.";
        }
    }
}

// Обработка удаления аватара
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_avatar'])) {
    // Проверка CSRF-токена
    if (!verify_csrf_token()) {
        $error = "Недействительный запрос. Попробуйте снова.";
    } else {
        if (!empty($user['avatar']) && file_exists($user['avatar'])) {
            unlink($user['avatar']);
        }
        $stmt = $pdo->prepare("UPDATE users SET avatar = NULL WHERE id = ?");
        $stmt->execute([$user_id]);
        header('Location: profile.php');
        exit;
    }
}

// Обработка обновления профиля (имя, город, сайт, о себе)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    // Проверка CSRF-токена
    if (!verify_csrf_token()) {
        $error = "Недействительный запрос. Попробуйте снова.";
    } else {
        $name = trim($_POST['name'] ?? '');
        $city = trim($_POST['city'] ?? '');
        $website = trim($_POST['website'] ?? '');
        $about = trim($_POST['about'] ?? '');

        try {
            $stmt = $pdo->prepare("UPDATE users SET name = ?, city = ?, website = ?, about = ? WHERE id = ?");
            $stmt->execute([$name, $city, $website, $about, $user_id]);
            header('Location: profile.php');
            exit;
        } catch (PDOException $e) {
            error_log("Ошибка обновления профиля: " . $e->getMessage());
            $error = "Произошла ошибка при обновлении профиля. Обратитесь к администратору.";
        }
    }
}

// Обработка изменения пароля
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    // Проверка CSRF-токена
    if (!verify_csrf_token()) {
        $error = "Недействительный запрос. Попробуйте снова.";
    } else {
        $current_password = trim($_POST['current_password'] ?? '');
        $new_password = trim($_POST['new_password'] ?? '');
        $confirm_password = trim($_POST['confirm_password'] ?? '');

        if (!empty($current_password) && !empty($new_password) && !empty($confirm_password)) {
            try {
                $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user_data = $stmt->fetch();

                if (password_verify($current_password, $user_data['password'])) {
                    if ($new_password === $confirm_password) {
                        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                        $stmt->execute([$hashed_password, $user_id]);
                        header('Location: profile.php?success=1');
                        exit;
                    } else {
                        $error = "Новый пароль и подтверждение не совпадают.";
                    }
                } else {
                    $error = "Текущий пароль неверен.";
                }
            } catch (PDOException $e) {
                error_log("Ошибка изменения пароля: " . $e->getMessage());
                $error = "Произошла ошибка при изменении пароля. Обратитесь к администратору.";
            }
        } else {
            $error = "Все поля должны быть заполнены.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="assets/js/script.js" defer></script>
    <title>Профиль — AstralForge</title>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-container">
            <?php if (isset($_SESSION['user_id'])): ?>
                <div class="navbar-user">
                    <a href="profile.php" class="navbar-avatar-link">
                        <?php 
                        $current_user = $pdo->prepare("SELECT avatar, login FROM users WHERE id = ?");
                        $current_user->execute([$_SESSION['user_id']]);
                        $current_user_data = $current_user->fetch();
                        ?>
                        <?php if (empty($current_user_data['avatar'])): ?>
                            <span class="default-avatar navbar-avatar">?</span>
                        <?php else: ?>
                            <img src="<?php echo htmlspecialchars($current_user_data['avatar']); ?>" alt="Ваш аватар" class="navbar-avatar" onerror="this.replaceWith(document.createElement('span').classList.add('default-avatar', 'navbar-avatar').textContent='?');">
                        <?php endif; ?>
                    </a>
                    <a href="profile.php" class="navbar-username"><?php echo htmlspecialchars($current_user_data['login']); ?></a>
                </div>
                <div class="navbar-links">
                    <a href="index.php">Главная</a>
                    <a href="user_list.php">Пользователи</a>
                    <a href="messages.php">
                        <span class="message-icon"></span>
                    </a>
                    <?php if (isset($_SESSION['is_admin']) && $_SESSION['is_admin']) echo '<a href="admin/index.php">Админ-панель</a>'; ?>
                    <a href="logout.php">Выход</a>
                </div>
            <?php endif; ?>
        </div>
    </nav>

    <div class="container">
        <h1 class="profile-title">Профиль пользователя: <?php echo htmlspecialchars($user['login']); ?></h1>
        
        <div class="profile-section">
            <h2>Аватар</h2>
            <div class="profile-header">
                <?php if (empty($user['avatar'])): ?>
                    <span class="default-avatar user-avatar">?</span>
                <?php else: ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Ваш аватар" class="user-avatar" onerror="this.replaceWith(document.createElement('span').classList.add('default-avatar', 'user-avatar').textContent='?');">
                <?php endif; ?>
            </div>
            <form method="POST" enctype="multipart/form-data" class="avatar-form">
                <input type="file" name="avatar" accept="image/jpeg,image/png,image/gif">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                <button type="submit" name="upload_avatar" class="chat-button">Загрузить аватар</button>
                <?php if (!empty($user['avatar'])): ?>
                    <button type="submit" name="delete_avatar" class="chat-button" style="background: linear-gradient(45deg, #e74c3c, #c0392b);">Удалить аватар</button>
                <?php endif; ?>
            </form>
        </div>

        <div class="profile-section">
            <h2>Редактировать профиль</h2>
            <form method="POST" class="profile-form">
                <div class="profile-details">
                    <div class="form-group">
                        <label for="name" class="info-label">Имя:</label>
                        <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($user['name'] ?: ''); ?>" class="info-value" placeholder="Введите имя">
                    </div>
                    <div class="form-group">
                        <label for="city" class="info-label">Город:</label>
                        <input type="text" id="city" name="city" value="<?php echo htmlspecialchars($user['city'] ?: ''); ?>" class="info-value" placeholder="Введите город">
                    </div>
                    <div class="form-group">
                        <label for="website" class="info-label">Веб-сайт:</label>
                        <input type="url" id="website" name="website" value="<?php echo htmlspecialchars($user['website'] ?: ''); ?>" class="info-value" placeholder="Введите URL сайта">
                    </div>
                    <div class="form-group">
                        <label for="about" class="info-label">О себе:</label>
                        <textarea id="about" name="about" class="info-value info-about" placeholder="Расскажите о себе"><?php echo htmlspecialchars($user['about'] ?: ''); ?></textarea>
                    </div>
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                    <button type="submit" name="update_profile" class="chat-button">Сохранить изменения</button>
                </div>
            </form>
        </div>

        <div class="profile-section">
            <h2>Изменить пароль</h2>
            <form method="POST" class="profile-form">
                <?php if (isset($error)): ?>
                    <p style="color: #e74c3c;"><?php echo htmlspecialchars($error); ?></p>
                <?php elseif (isset($_GET['success']) && $_GET['success'] == 1): ?>
                    <p style="color: #2ecc71;">Пароль успешно изменён!</p>
                <?php endif; ?>
                <div class="profile-details">
                    <div class="form-group">
                        <label for="current_password" class="info-label">Текущий пароль:</label>
                        <input type="password" id="current_password" name="current_password" class="info-value" placeholder="Введите текущий пароль" required>
                    </div>
                    <div class="form-group">
                        <label for="new_password" class="info-label">Новый пароль:</label>
                        <input type="password" id="new_password" name="new_password" class="info-value" placeholder="Введите новый пароль" required>
                    </div>
                    <div class="form-group">
                        <label for="confirm_password" class="info-label">Подтвердите новый пароль:</label>
                        <input type="password" id="confirm_password" name="confirm_password" class="info-value" placeholder="Повторите новый пароль" required>
                    </div>
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                    <button type="submit" name="change_password" class="chat-button">Изменить пароль</button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>