<?php

namespace Akh\Typograf\Tests\Rule\Quote;

use Akh\Typograf\Rule\AbstractRule;
use Akh\Typograf\Rule\Quote\Quote;
use Akh\Typograf\Tests\Rule\RuleTestCase;

/**
 * @internal
 */
class QuoteTest extends RuleTestCase
{
    public function getRule(): AbstractRule
    {
        return new Quote();
    }

    /**
     * @return string[][]
     */
    public function dataProvider(): array
    {
        return [
            [
                'специально для "клапана на 3/4" или 1/2" (наружная резьба)" нужно дополнительно',
                'специально для &laquo;клапана на 3/4″ или 1/2″ (наружная резьба)&raquo; нужно дополнительно',
            ],
            [
                'Вот у вас "Мой спутник, "это "не "сочинение" это" хорошо, потому" это хорошо, потому что не выдумано."',
                'Вот у вас &laquo;Мой спутник, &laquo;это &laquo;не &laquo;сочинение&raquo; это&raquo; хорошо, потому&raquo; это хорошо, потому что не выдумано.&raquo;',
            ],
            [
                'Вот у вас "Мой спутник" – это не сочинение, это хорошо, потому что не выдумано.',
                'Вот у вас &laquo;Мой спутник&raquo; – это не сочинение, это хорошо, потому что не выдумано.',
            ],
            [
                '««Цыганы» мои не продаются вовсе»',
                '&laquo;&bdquo;Цыганы&ldquo; мои не продаются вовсе&raquo;',
            ],
            [
                '"Пример"',
                '&laquo;Пример&raquo;',
            ],
            [
                "\"Пример\"\n\"Пример\"",
                "&laquo;Пример&raquo;\n&laquo;Пример&raquo;",
            ],
            [
                '<p>"Пример"</p>',
                '<p>&laquo;Пример&raquo;</p>',
            ],
            [
                'ОАО "Пример"',
                'ОАО &laquo;Пример&raquo;',
            ],
            [
                'В "самом "добром" кино" Мамы. В "самом "добром" кино" Мамы',
                'В &laquo;самом &bdquo;добром&ldquo; кино&raquo; Мамы. В &laquo;самом &bdquo;добром&ldquo; кино&raquo; Мамы',
            ],
            [
                'В самом добром кино “Мамы”, в молодежном триллере “Закрытая школа” на СТС. А еще на сцене театра им. Вл. Маяковского в спектакле “Не все коту масленица”.',
                'В самом добром кино &laquo;Мамы&raquo;, в молодежном триллере &laquo;Закрытая школа&raquo; на СТС. А еще на сцене театра им. Вл. Маяковского в спектакле &laquo;Не все коту масленица&raquo;.',
            ],
            [
                'В самом добром кино “Мамы”, в молодежном триллере “Закрытая школа” на СТС',
                'В самом добром кино &laquo;Мамы&raquo;, в молодежном триллере &laquo;Закрытая школа&raquo; на СТС',
            ],
            [
                'В самом добром кино “Мамы, в молодежном триллере “Закрытая школа” на СТС"',
                'В самом добром кино &laquo;Мамы, в молодежном триллере &bdquo;Закрытая школа&ldquo; на СТС&raquo;',
            ],
            [
                "<p>\"Я всегда с гордостью носил военную форму...</p>\n\n<p>...Я переживал очень тяжёлую депрессию.\"</p>",
                "<p>&laquo;Я всегда с гордостью носил военную форму...</p>\n\n<p>...Я переживал очень тяжёлую депрессию.&raquo;</p>",
            ],
            [
                'Печорин писал: "Я не помню утра более голубого и свежего!"',
                'Печорин писал: &laquo;Я не помню утра более голубого и свежего!&raquo;',
            ],
            [
                'Печорин признавался: "Я иногда себя презираю..."',
                'Печорин признавался: &laquo;Я иногда себя презираю...&raquo;',
            ],
            [
                'Печорин спрашивает: "И зачем было судьбе кинуть меня в мирный круг честных контрабандистов?"',
                'Печорин спрашивает: &laquo;И зачем было судьбе кинуть меня в мирный круг честных контрабандистов?&raquo;',
            ],
            [
                'Печорин размышляет: "…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие..."',
                'Печорин размышляет: &laquo;…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие...&raquo;',
            ],
            [
                'Печорин размышляет: "…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие…"',
                'Печорин размышляет: &laquo;…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие…&raquo;',
            ],
            [
                "Печорин размышляет: \"…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие…\"\n\n",
                "Печорин размышляет: &laquo;…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие…&raquo;\n\n",
            ],
            [
                "Печорин размышляет: \"…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие…\"\n\nПечорин...",
                "Печорин размышляет: &laquo;…зачем было судьбе кинуть меня в мирный круг честных контрабандистов? Как камень, брошенный в гладкий источник, я встревожил их спокойствие…&raquo;\n\nПечорин...",
            ],
            [
                'Лермонтов восклицает в предисловии, что это "старая и жалкая шутка!"',
                'Лермонтов восклицает в предисловии, что это &laquo;старая и жалкая шутка!&raquo;',
            ],
            [
                '"Лермонтов восклицает в "предисловии", что это "старая и жалкая шутка!""',
                '&laquo;Лермонтов восклицает в &bdquo;предисловии&ldquo;, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo;',
            ],
            [
                '"Лермонтов восклицает в "предисловии", что это "старая и жалкая шутка!"" "Лермонтов восклицает в "предисловии", что это "старая и жалкая шутка!""',
                '&laquo;Лермонтов восклицает в &bdquo;предисловии&ldquo;, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo; &laquo;Лермонтов восклицает в &bdquo;предисловии&ldquo;, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo;',
            ],
            [
                '"Лермонтов восклицает в предисловии, что это "старая и жалкая шутка!"" "Лермонтов восклицает в предисловии, что это "старая и жалкая шутка!""',
                '&laquo;Лермонтов восклицает в предисловии, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo; &laquo;Лермонтов восклицает в предисловии, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo;',
            ],
            [
                '"Лермонтов восклицает в предисловии, что это "старая и жалкая шутка!"" "Лермонтов восклицает в предисловии, что это "старая и жалкая шутка!"" "Лермонтов восклицает в предисловии, что это "старая и жалкая шутка!""',
                '&laquo;Лермонтов восклицает в предисловии, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo; &laquo;Лермонтов восклицает в предисловии, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo; &laquo;Лермонтов восклицает в предисловии, что это &bdquo;старая и жалкая шутка!&ldquo;&raquo;',
            ],
            [
                '<p>"Энергия соблазна: "от внутреннего" к внешнему".</p>        <p>"Энергия соблазна: "от внутреннего" к внешнему".</p>',
                '<p>&laquo;Энергия соблазна: &bdquo;от внутреннего&ldquo; к внешнему&raquo;.</p>        <p>&laquo;Энергия соблазна: &bdquo;от внутреннего&ldquo; к внешнему&raquo;.</p>',
            ],
            [
                "<p>\"Энергия\nсоблазна: \"от\nвнутреннего\" к\nвнешнему\".</p>        <p>\"Энергия\nсоблазна: \"от\nвнутреннего\" к\nвнешнему\".</p>",
                "<p>&laquo;Энергия\nсоблазна: &bdquo;от\nвнутреннего&ldquo; к\nвнешнему&raquo;.</p>        <p>&laquo;Энергия\nсоблазна: &bdquo;от\nвнутреннего&ldquo; к\nвнешнему&raquo;.</p>",
            ],
            [
                "\"Полотенцесушители из нержавеющей стали\"\n\nПолотенцесушитель из черного металла, сделанные из нержавеющей стали, очень хорошо подходят к использованию в наших условиях. Снаружи они могут иметь полированную, матовую, или даже окрашенную поверхность. Модели с окрашенной поверхностью обычно стоят меньше других. Еще один плюс окрашенных полотенцесушителей — возможность разместить их в любом интерьере благодаря широкой цветовой гамме.",
                "&laquo;Полотенцесушители из нержавеющей стали&raquo;\n\nПолотенцесушитель из черного металла, сделанные из нержавеющей стали, очень хорошо подходят к использованию в наших условиях. Снаружи они могут иметь полированную, матовую, или даже окрашенную поверхность. Модели с окрашенной поверхностью обычно стоят меньше других. Еще один плюс окрашенных полотенцесушителей — возможность разместить их в любом интерьере благодаря широкой цветовой гамме.",
            ],
            [
                '<i>"Энергия соблазна".</i>',
                '<i>&laquo;Энергия соблазна&raquo;.</i>',
            ],
            [
                '<i>"Энергия соблазна".</i> <i>"Энергия соблазна".</i>',
                '<i>&laquo;Энергия соблазна&raquo;.</i> <i>&laquo;Энергия соблазна&raquo;.</i>',
            ],
            [
                '"<i>Энергия соблазна</i>".',
                '&laquo;<i>Энергия соблазна</i>&raquo;.',
            ],
            [
                '"<i>Энергия соблазна</i>". "<i>Энергия соблазна</i>".',
                '&laquo;<i>Энергия соблазна</i>&raquo;. &laquo;<i>Энергия соблазна</i>&raquo;.',
            ],
            [
                '"<i>Энергия соблазна"</i>.',
                '&laquo;<i>Энергия соблазна&raquo;</i>.',
            ],
            [
                '"<i>Энергия соблазна"</i>. "<i>Энергия соблазна"</i>.',
                '&laquo;<i>Энергия соблазна&raquo;</i>. &laquo;<i>Энергия соблазна&raquo;</i>.',
            ],
            [
                '"<i>Екатеринбург лучше Стамбула, однозначно, а&nbsp;люди здесь добрее, чем в&nbsp;Москве и&nbsp;Питере"</i>, &nbsp;- рассказывает он&nbsp;со&nbsp;знанием дела: во&nbsp;время подготовки книги об&nbsp;уральской столице Аслиддин прожил по&nbsp;две недели и&nbsp;в&nbsp;Белокаменной, и&nbsp;в&nbsp;Северной Пальмире. &nbsp;"<i>Все оплачивают спонсоры&nbsp;-; бизнесмены из&nbsp;Таджикистана, которые заинтересовались моими работами. Они-то и&nbsp;договаривались с&nbsp;издательством"</i>, &nbsp;-; объясняет&nbsp;строитель-историк.&nbsp;Из&nbsp;пятисот отпечатанных экземпляров автору достанется 15. Все остальные пойдут на&nbsp;магазинные прилавки и&nbsp;в&nbsp;Центральную библиотеку Таджикистана, а&nbsp;также самим спонсорам. Размеры гонорара, который дойдет до&nbsp;Аслиддина, не&nbsp;позволят уволиться со&nbsp;стройки. &nbsp;<i>"Да я&nbsp;рад уже тому, что мои книги свет увидели, их&nbsp;читают, отзывы делают", &nbsp;</i>- рассказывает писатель.',
                '&laquo;<i>Екатеринбург лучше Стамбула, однозначно, а&nbsp;люди здесь добрее, чем в&nbsp;Москве и&nbsp;Питере&raquo;</i>, &nbsp;- рассказывает он&nbsp;со&nbsp;знанием дела: во&nbsp;время подготовки книги об&nbsp;уральской столице Аслиддин прожил по&nbsp;две недели и&nbsp;в&nbsp;Белокаменной, и&nbsp;в&nbsp;Северной Пальмире. &nbsp;&laquo;<i>Все оплачивают спонсоры&nbsp;-; бизнесмены из&nbsp;Таджикистана, которые заинтересовались моими работами. Они-то и&nbsp;договаривались с&nbsp;издательством&raquo;</i>, &nbsp;-; объясняет&nbsp;строитель-историк.&nbsp;Из&nbsp;пятисот отпечатанных экземпляров автору достанется 15. Все остальные пойдут на&nbsp;магазинные прилавки и&nbsp;в&nbsp;Центральную библиотеку Таджикистана, а&nbsp;также самим спонсорам. Размеры гонорара, который дойдет до&nbsp;Аслиддина, не&nbsp;позволят уволиться со&nbsp;стройки. &nbsp;<i>&laquo;Да я&nbsp;рад уже тому, что мои книги свет увидели, их&nbsp;читают, отзывы делают&raquo;, &nbsp;</i>- рассказывает писатель.',
            ],
            [
                '"',
                '&quot;',
            ],
            [
                '"Газета',
                '&laquo;Газета',
            ],
            [
                '" Газета',
                '&quot; Газета',
            ],
            [
                '"Иннопром". "Иннопром". "Синий "мужчина" знак".',
                '&laquo;Иннопром&raquo;. &laquo;Иннопром&raquo;. &laquo;Синий &bdquo;мужчина&ldquo; знак&raquo;.',
            ],
            [
                'М. М. Бахтин писал: "Тришатов рассказывает подростку о своей любви к музыке и развивает перед ним замысел оперы: "Послушайте, любите вы музыку? Я ужасно люблю... Если бы я сочинял оперу, то, знаете, я бы взял сюжет из "Фауста". Я очень люблю эту тему"".',
                'М. М. Бахтин писал: &laquo;Тришатов рассказывает подростку о своей любви к музыке и развивает перед ним замысел оперы: &bdquo;Послушайте, любите вы музыку? Я ужасно люблю... Если бы я сочинял оперу, то, знаете, я бы взял сюжет из ‚Фауста‘. Я очень люблю эту тему&ldquo;&raquo;.',
            ],
            [
                'по произведению Достоевского &quot;Преступление и наказание&quot;в театре Моссовета',
                'по произведению Достоевского &laquo;Преступление и наказание&quot;в театре Моссовета',
            ],
            [
                'Из всей нашей культурной программы самое сильное впечатление на меня произвела постановка &quot;Р.Р.Р.&quot; по произведению Достоевского &quot;Преступление и наказание&quot;в театре Моссовета.' .
                "Она буквально влюбила меня в театр.\n&quot;Мы напихиваем в детей ненужное барахло&quot;. 5 радикальных тезисов Германа Грефа об образовании",

                'Из всей нашей культурной программы самое сильное впечатление на меня произвела постановка &laquo;Р.Р.Р.&raquo; по произведению Достоевского &laquo;Преступление и наказание&quot;в театре Моссовета.' .
                "Она буквально влюбила меня в театр.\n&laquo;Мы напихиваем в детей ненужное барахло&raquo;. 5 радикальных тезисов Германа Грефа об образовании",
            ],
            [
                'Из всей нашей культурной программы самое сильное впечатление на меня произвела постановка &quot;Р.Р.Р.&quot; по произведению Достоевского &quot;Преступление и наказание&quot;в театре Моссовета.' .
                "Она буквально влюбила меня в театр.\n&quot;Мы напихиваем в детей ненужное барахло&quot;. 5 радикальных тезисов Германа Грефа об образовании\n" .
                'Из всей нашей культурной программы самое сильное впечатление на меня произвела постановка &quot;Р.Р.Р.&quot; по произведению Достоевского &quot;Преступление и наказание&quot;в театре Моссовета.' .
                "Она буквально влюбила меня в театр.\n&quot;Мы напихиваем в детей ненужное барахло&quot;. 5 радикальных тезисов Германа Грефа об образовании",

                'Из всей нашей культурной программы самое сильное впечатление на меня произвела постановка &laquo;Р.Р.Р.&raquo; по произведению Достоевского &laquo;Преступление и наказание&quot;в театре Моссовета.' .
                "Она буквально влюбила меня в театр.\n&laquo;Мы напихиваем в детей ненужное барахло&raquo;. 5 радикальных тезисов Германа Грефа об образовании\n" .
                'Из всей нашей культурной программы самое сильное впечатление на меня произвела постановка &laquo;Р.Р.Р.&raquo; по произведению Достоевского &laquo;Преступление и наказание&quot;в театре Моссовета.' .
                "Она буквально влюбила меня в театр.\n&laquo;Мы напихиваем в детей ненужное барахло&raquo;. 5 радикальных тезисов Германа Грефа об образовании",
            ],
            [
                '<p>«Куда ты ведёшь нас?.. не видно ни зги! —</p>',
                '<p>&laquo;Куда ты ведёшь нас?.. не видно ни зги! —</p>',
            ],
            [
                '“слово слово “слово” слово”',
                '&laquo;слово слово &bdquo;слово&ldquo; слово&raquo;',
            ],
            [
                'Движение перекроют из‑за матча на«ВЭБ Арене» 7 октября в Москве‍',
                'Движение перекроют из‑за матча на&laquo;ВЭБ Арене&raquo; 7 октября в Москве‍',
            ],
            [
                '["история наоборот"]',
                '[&laquo;история наоборот&raquo;]',
            ],
            [
                '[»история наоборот»]',
                '[&laquo;история наоборот&raquo;]',
            ],
            [
                '&raquo;история наоборот&raquo;',
                '&laquo;история наоборот&raquo;',
            ],
            [
                'Линза 1,5"',
                'Линза 1,5″',
            ],
            [
                'Линза 1.5"',
                'Линза 1.5″',
            ],
        ];
    }

    public function toDo(): void
    {
        $arTests = [
            [
                "<p>\"Я всегда с гордостью носил военную форму...</p>\n\n<p>...Я переживал очень тяжёлую депрессию. \"</p>",
                "<p>&laquo;Я всегда с гордостью носил военную форму...</p>\n\n<p>...Я переживал очень тяжёлую депрессию. &raquo;</p>",
            ],
            [
                '“ слово слово “слово” слово”',
                '" слово слово &laquo;слово&raquo; слово"',
            ],
        ];
    }

    public function testInchDisabled(): void
    {
        $arTests = [
            [
                'специально для "клапана на 3/4" или 1/2" (наружная резьба)" нужно дополнительно',
                'специально для &laquo;клапана на 3/4&raquo; или 1/2&raquo; (наружная резьба)&raquo; нужно дополнительно',
            ],
        ];

        foreach ($arTests as $arTest) {
            $rule = new Quote();
            $rule->setSetting('inch', false);
            $test = $rule->Handler($arTest[0]);
            $this->assertSame($test, $arTest[1]);
        }
    }
}
